import { ChatId, ContactId, MessageId } from "./aliases";
import { Message } from "./message";

export interface CustomProduct {
    /**
     * The main title of the product. E.g:
     * `BAVARIA — 35 SPORTS CRUISER (2006)`
     */
    name: string;
    /**
     * The description of the product. This shows right under the price so it is useful for subscriptions/rentals. E.g:
     * 
     * `(per day)\n\nCome and have a fantastic sailing adventure aboard our boat. \nShe is a Bavaria 35 sports cruiser and is powered by 2 economical Volvo D6’s with Bravo 2 outdrives as well as a bow thruster. This Makes maneuvering very easy. She can accommodate up to 8 people for day charters and for overnight charters she can accommodate 4 in comfort in 2 cabins.`
     */
    description: string;
    /**
     * The price amount multiplied by 1000. For example, for something costing `825` units of currency:
     * `825000`
     */
    priceAmount1000: number;
    /**
     * The [**ISO 4217**](https://en.wikipedia.org/wiki/ISO_4217) 3 letter currency code. E.g (Swedish krona)
     * `SEK`
     */
    currency: string;
    /**
     * The URL of the product.
     * 
     * NOTE: At the moment, the URL DOES NOT WORK. It shows up for the recipient but they will not be able to click it. As a rememdy, it is added as a reply to the product message.
     */
    url?: string
}

export interface CartItem {
    /**
     * Product ID
     */
     id: string,
     /**
      * Product name
      */
     name: string,
     /**
      * Amount of this item in the cart
      */
     qty: number,
     thumbnailId: string,
     /**
      * URL to .enc file of the thumbnail. Just change the filetype to .jpg to view the thumbnail
      */
     thumbnailUrl: string,
}


export interface Product {
    /**
     * Product ID
     */
    id: string;
    /**
     * `true` if the product is hidden from public view.
     */
    isHidden?: boolean;
    /**
     * The id of the catalog in which this product is located.
     */
    catalogWid?: string;
    /**
     * The URL of the product.
     */
    url ?: string;
    /**
     * The name of the product.
     */
    name ?: string;
    /**
     * The description of the product.
     */
    description ?: string;
    /**
     * The availiable quantity of this product.
     * @default "unknown"`
     */
    availability ?: number | "unknown";
    /**
     * The review status of the product
     */
    reviewStatus ?: "NO_REVIEW" | "PENDING" | "REJECTED" | "APPROVED" | "OUTDATED",
    /**
    * The url of the main image of the product.
    * 
    * NOTE: If downloading manually, the filetype must be changed to .jpg to view the image.
    */
    imageCdnUrl ?: string,
    /**
    * The number of images of the product.
    */
    imageCount ?: number,
    /**
     * Array of URLs of the other images of the product. Does not include the main image.
     */
    additionalImageCdnUrl ?: string[],
    /**
     * The price of the product in 1000 units. 
     */
    priceAmount1000 ?: number,
    /**
     * The custom id of the product.
     */
    retailerId ?: string,
    /**
     * The timestamp when the product was created / 1000
     */
    t ?: number,
    /**
     * The [**ISO 4217**](https://en.wikipedia.org/wiki/ISO_4217) 3 letter currency code. E.g (Swedish krona)
     * `SEK`
     */
    currency: string,
}

export interface Order {
    /**
     * Order ID
     */
    id: string,
    /**
     * epoch ts divided by 1000
     */
    createdAt: number,
    /**
     * The [**ISO 4217**](https://en.wikipedia.org/wiki/ISO_4217) 3 letter currency code. E.g (Swedish krona)
     * `SEK`
     */
    currency: string,
    /**
     * An array of items in the cart
     */
    products: CartItem[],
    sellerJid: string,
    subtotal: `${number}`,
    total:  `${number}`,
    /**
     * The message object associated with the order. Only populated in `onOrder` callback.
     */
    message ?: Message
}
